#ifndef __CMidiEventList__
#define __CMidiEventList__

//	===========================================================================

#include "CMidiEvent.hpp"
using Exponent::Midi::CMidiEvent;
using Exponent::Basics::CCountedObject;

//	===========================================================================

namespace Exponent
{
	namespace Midi
	{
		/**
		 * @class CMidiEventList CMidiEventList.hpp
		 * @brief Provides handling for a buffers worth of midi
		 *
		 * Can be filled at the start of a buffer with the midi information\n
		 * and can then provides extra information about those buffers\n
		 * This is an optimised version of a midi event list, used and designed for real time application\n
		 * (such as inside a VST plugin). It is not designed for long pieces of midi data, which will break\n
		 * the size constraints of the event list (@see CMIDI_EVENT_LIST_MAX_EVENTS).\n
		 * Although it does handle midi channel its other handling is somewhat dubious and is definately not meant\n
		 * for use when reading or writing midi files.
		 *
		 * @see CMidiFile
		 * @see CMidiSequence
		 * @see CMidiTrack
		 *
		 * @date 29/08/2004
		 * @author Paul Chana
		 * @version 1.0.0 Initial version
		 * @version 1.0.1 Added the getIterationObject function
		 *
		 * @note All contents of this source code are copyright 2005 Exp Digital Uk.\n
		 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy\n
		 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
		 * All content is the Intellectual property of Exp Digital Uk.\n
		 * Certain sections of this code may come from other sources. They are credited where applicable.\n
		 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
		 *
		 * $Id: CMidiEventList.hpp,v 1.4 2007/02/08 21:08:09 paul Exp $
		 */
		class CMidiEventList : public CCountedObject
		{
			/** @cond */
			EXPONENT_CLASS_DECLARATION;
			/** @endcond */

//	===========================================================================

		public:

//	===========================================================================

			const static long CMIDI_EVENT_LIST_MAX_EVENTS = 128;		/**< Maximum events per buffer */

//	===========================================================================

			/**
			 * Construction
			 */
			CMidiEventList();

			/**
			 * Destruction
			 */
			virtual ~CMidiEventList();

//	===========================================================================

			/**
			 * Get an event at a location
			 * @param index The index of the event
			 * @retval const CMidiEvent* The event or null if index range error
			 */
			const CMidiEvent *getEventAtIndex(const long index) const;

			/**
			 * Get the time different between the selected node and the next node. Assumed sorted list
			 * @retval long The delta difference
			 */
			long getTimeDeltaDifference() const;

//	===========================================================================

			/**
			 * Set the midi channel, events not on this midi channel are ignored
			 * @param midiChannel The midi channel to use
			 */
			void setMidiChannel(const long midiChannel) { m_midiChannel = midiChannel; }

			/**
			 * Get the midi channel
			 * @retval long The midi channel
			 */
			long getMidiChannel() const { return m_midiChannel; }

			/**
			 * Get the number of events
			 * @retval long The number of events for this buffer
			 */
			long getNumberOfEvents() { return m_numberOfEvents; }

//	===========================================================================

			/**
			 * Get the next event
			 * @retval const CMidiEvent* The next midi event or NULL if no more midi events
			 */
			const CMidiEvent *getNextEvent() const;

			/**
			 * Increment to next event
			 */
			void moveToNextEvent();

			/**
			 * Are there more events
			 * @retval bool True if there are more events
			 */
			bool moreEvents() const;

			/**
			 * Clear all events
			 */
			void clearEvents();

//	===========================================================================

			/**
			 * Get a description of the object
			 * @param string On return is filled with the description
			 * @param size The size of the stirng
			 */
			virtual void getObjectDescription(char *string, const long size) const;

//	===========================================================================

		protected:

//	===========================================================================

			CMidiEvent m_list[CMIDI_EVENT_LIST_MAX_EVENTS];			/**< The midi event list */
			long m_midiChannel;										/**< Channel we will respond to */
			long m_numberOfEvents;									/**< How many events do we have */
			long m_currentEvent;									/**< The current event */

		};
	}
}
#endif	// End of CMidiEventList.hpp